<?php

/**
 * Register admin dashboard
 *
 * @return void
 */
function toman_escrow_admin_register()
{
    // Menu slug
    $mainMenuSlug = 'toman-escrow';

    // Define main menu
    toman_escrow_admin_add_main_menu($mainMenuSlug);

    // Define orders list submenu
    toman_escrow_admin_add_orders_page($mainMenuSlug);

    // Define shipment form submenu
    toman_escrow_admin_add_shipment_form_page($mainMenuSlug);

    // Add registration link
    toman_escrow_admin_add_toman_registration_link($mainMenuSlug);

    // Load CSS files
    toman_escrow_admin_enqueue_assets();
}

add_action('admin_menu', 'toman_escrow_admin_register');

/**
 * Add main menu
 *
 * @param string $mainMenuSlug
 * @return void
 */
function toman_escrow_admin_add_main_menu(string $mainMenuSlug)
{
    $gateway = new WC_Toman_Safe_Pay();

    $notification_count = $gateway->get_funded_new_deal_count_in_toman(); // <- here you should get correct count of forms submitted since last visit
    add_menu_page(
        'پرداخت امن تومن',
        $notification_count ? sprintf('پرداخت امن تومن <span class="awaiting-mod">%d</span>', $notification_count) : 'پرداخت امن تومن',
        'manage_options',
        $mainMenuSlug,
        'toman_escrow_admin_render_config_page',
        'dashicons-money-alt',
        10
    );
}

/**
 * Add orders page
 *
 * @param string $mainMenuSlug
 * @return void
 */
function toman_escrow_admin_add_orders_page(string $mainMenuSlug)
{
    add_submenu_page($mainMenuSlug,
        'لیست سفارشات',
        'لیست سفارشات',
        'manage_options',
        $mainMenuSlug . '-orders',
        'toman_escrow_admin_render_orders_page'
    );
}

/**
 * Add shipment page
 *
 * @param string $mainMenuSlug
 * @return void
 */
function toman_escrow_admin_add_shipment_form_page(string $mainMenuSlug)
{
    add_submenu_page($mainMenuSlug,
        'ثبت جزییات ارسال',
        'ثبت جزییات ارسال',
        'manage_options',
        $mainMenuSlug . '-shipment',
        'toman_escrow_admin_render_shipment_page'
    );
}

/**
 * Add Toman Escrow registration link
 *
 * @param string $mainMenuSlug
 * @return void
 */
function toman_escrow_admin_add_toman_registration_link(string $mainMenuSlug)
{
    add_submenu_page($mainMenuSlug,
        'ثبت‌نام در پرداخت امن تومن',
        'ثبت‌نام در پرداخت امن تومن',
        'manage_options',
        TOMAN_ESCROW_REGISTRATION_LINK
    );
}

/**
 * Redirect to Toman config page (In WooCommerce)
 *
 * @return void
 */
function toman_escrow_admin_render_config_page()
{
    wp_redirect(admin_url('admin.php?page=wc-settings&tab=checkout&section=wc_toman_safe_pay'));
    exit;
}

function get_ordering_from_key($key)
{
    $toman_ordering = array(
        "id" => array(
            "ordery_by" => "id",
            "direction" => "DESC",
            "query_reverse_field_name" => "-id",
            "query_field_name" => "id",
        ),
        "-id" => array(
            "ordery_by" => "id",
            "direction" => "ASC",
            "query_reverse_field_name" => "id",
            "query_field_name" => "-id",
        ),
        "date" => array(
            "ordery_by" => "date_created",
            "direction" => "DESC",
            "query_reverse_field_name" => "-date",
            "query_field_name" => "date",
        ),
        "-date" => array(
            "ordery_by" => "date_created",
            "direction" => "ASC",
            "query_reverse_field_name" => "date",
            "query_field_name" => "-date",
        ),
    );
    $default_ordering = array(
        "ordery_by" => "id",
        "direction" => "DESC",
        "query_reverse_field_name" => "-id",
        "query_field_name" => "id",
    );
    if (array_key_exists($key, $toman_ordering)) {
        return $toman_ordering[$key];
    } else {
        return $default_ordering;
    }
}

/**
 * Render orders page
 *
 * @return void
 */
function toman_escrow_admin_render_orders_page()
{
    $gateway = new WC_Toman_Safe_Pay();
    $query_params = $_GET;// phpcs:ignore

    if (isset($query_params['trace_number']) && isset($query_params['order_id'])) {
        $orderShouldBeCompleted = $gateway->verify_order_by_toman_api($query_params['trace_number']);

        // Complete order
        if ($orderShouldBeCompleted) {
            $order = new WC_Order($query_params['order_id']);
            $gateway->complete_order($order, $query_params['trace_number'], false);
        }
        unset($query_params['trace_number']);
        unset($query_params['order_id']);
        $redirect_url = admin_url('admin.php?page=toman-escrow-orders');
        if (!empty($query_params)) {
            $redirect_url = add_query_arg($query_params, $redirect_url);
        }
        wp_redirect($redirect_url);
        exit;
    }
    if (isset($query_params['order_field'])) {
        $ordering = get_ordering_from_key($query_params['order_field']);
    } else {
        $ordering = get_ordering_from_key("non-existent-key");
    }
    $pagenum = isset($query_params['pagenum']) ? absint($query_params['pagenum']) : 1;
    $limit = 10;
    $query_params['pagenum'] = $pagenum;
    $ordering_links = array(
        "id" => add_query_arg("order_field", "id" == $ordering["query_field_name"] ? $ordering["query_reverse_field_name"] : "id", add_query_arg($query_params)),
        "date" => add_query_arg("order_field", "date" == $ordering["query_field_name"] ? $ordering["query_reverse_field_name"] : "date", add_query_arg($query_params)),
    );
    $total = count(
        wc_get_orders(
            array(
                "payment_method" => "WC_Toman_Safe_Pay",
                "limit" => -1,
                'return' => 'ids',  // Return only the IDs
            )
        )
    );
    $args = array(
        'payment_method' => "WC_Toman_Safe_Pay",
        'orderby' => $ordering['ordery_by'],
        'order' => $ordering['direction'],
        "limit" => $limit,
        "page" => $pagenum,
    );
    $orders = wc_get_orders($args);
    $gateway->sync_last_uncompleted_orders_statuses_by_toman_api($orders);

    $num_of_pages = ceil($total / $limit);
    $query_params['pagenum'] = '%#%';
    $page_links = paginate_links(array(
        'base' => add_query_arg($query_params),
        'format' => '',
        'prev_text' => __('&laquo;', 'text-domain'),
        'next_text' => __('&raquo;', 'text-domain'),
        'total' => $num_of_pages,
        'current' => $pagenum
    ));
    // Define order currency
    $currency = '';
    if (count($orders) > 0) {
        $currency = get_woocommerce_currency_symbol($orders[0]->get_currency());
    }
    // Translate statuses
    $translatedStatuses = [
        'pending' => __('Pending', 'toman-escrow-for-woocommerce'),
        'processing' => __('Processing', 'toman-escrow-for-woocommerce'),
        'on-hold' => __('Hold', 'toman-escrow-for-woocommerce'),
        'completed' => __('Completed', 'toman-escrow-for-woocommerce'),
        'cancelled' => __('Cancelled', 'toman-escrow-for-woocommerce'),
        'refunded' => __('Refunded', 'toman-escrow-for-woocommerce'),
        'failed' => __('Failed', 'toman-escrow-for-woocommerce'),
    ];

    $file = TOMAN_ESCROW_PLUGIN_DIR . "views/admin/orders.html";
    if (file_exists($file)) {
        require $file;
    }
}

/**
 * Render shipment page
 *
 * @return void
 */
function toman_escrow_admin_render_shipment_page()
{
    // Get last order IDs
    if (!isset($_GET["order_id"])) {// phpcs:ignore
        // Return HTTP 403
        status_header(403);
        nocache_headers();
        echo '<h1>403 Forbidden</h1>';
        echo '<p>You do not have permission to access this page.</p>';
        exit;
    }
    $order_id = intval($_GET["order_id"]);// phpcs:ignore
    $order = wc_get_order($order_id);
    if ($order && $order->get_payment_method() == "WC_Toman_Safe_Pay" && in_array($order->get_status(), array("on-hold", "processing"))) {
        // Render page
        $file = TOMAN_ESCROW_PLUGIN_DIR . "views/admin/shipment.html";
        if (file_exists($file)) {
            require $file;
        }
    } else {
        // Return HTTP 403
        status_header(403);
        nocache_headers();
        echo '<h1>403 Forbidden</h1>';
        echo '<p>You do not have permission to access this page.</p>';
        exit;
    }
}

// TODO: Check why this doesn't work
// TODO: Work on admin panel appearance
/**
 * Enqueue styles
 *
 * @return void
 */
function toman_escrow_admin_enqueue_assets()
{
    wp_enqueue_style('toman-admin-dashboard', TOMAN_ESCROW_PLUGIN_DIR . "assets/css/style.css", false, '1.0.0', 'all');
}

function get_admin_url_for_verification($q_params, $trace_number, $order_id)
{
    $q_params['trace_number'] = $trace_number;
    $q_params['order_id'] = $order_id;
    $redirect_url = admin_url('admin.php?page=toman-escrow-orders');
    if (!empty($q_params)) {
        $redirect_url = add_query_arg($q_params, $redirect_url);
    }
    return $redirect_url;
}

function get_shipment_url($query_params, $order_id)
{
    $query_params["order_id"] = $order_id;
    $query_params["page"] = "toman-escrow-shipment";
    $redirect_url = admin_url('admin.php');
    $redirect_url = add_query_arg($query_params, $redirect_url);
    return $redirect_url;
}