<?php

// Don't allow to access to this plugin directly
if (!defined('ABSPATH')) {
    exit;
}

/**
 * Add Toman Escrow payment gateway to WooCommerce payment gateways
 *
 * @param array $gateways
 * @return array
 */
function toman_escrow_add_gateway_class(array $gateways): array
{
    $gateways[] = TOMAN_ESCROW_PAYMENT_GATEWAY_ID;

    return $gateways;
}

add_filter('woocommerce_payment_gateways', 'toman_escrow_add_gateway_class');

/**
 * Define Toman Escrow plugin
 *
 * @return void
 */
function toman_escrow_define_plugin()
{
    // Check plugin activation requirements
    toman_escrow_check_plugin_can_activate();

    // Define Toman safe pay gateway class
    class WC_Toman_Safe_Pay extends WC_Payment_Gateway
    {
        /**
         * The payment status of a success payment
         */
        const SUCCESS_PAYMENT_STATUS = 'funded';

        /**
         * The sub_status of a success payment
         */
        const SUCCESS_PAYMENT_SUB_STATUS = 'new';

        /**
         * Toman status (By state and sub_state): Pending
         */
        const TOMAN_STATUS_PENDING = 'pending';

        /**
         * Toman status (By state and sub_state): Processing
         */
        const TOMAN_STATUS_PROCESSING = 'processing';

        /**
         * Toman status (By state and sub_state): Completed
         */
        const TOMAN_STATUS_COMPLETED = 'completed';

        /**
         * Toman status (By state and sub_state): Cancelled
         */
        const TOMAN_STATUS_CANCELLED = 'cancelled';

        /**
         * Toman status (By state and sub_state): Refunded
         */
        const TOMAN_STATUS_REFUNDED = 'refunded';

        /**
         * Toman status (By state and sub_state): Failed
         */
        const TOMAN_STATUS_FAILED = 'failed';

        /**
         * Toman status (By state and sub_state): On Hold
         */
        const TOMAN_STATUS_ON_HOLD = 'on-hold';

        /**
         * Shop slug
         *
         * @var string
         */
        public $shopSlug;

        /**
         * Authentication code
         *
         * @var string
         */
        public $authenticationCode;

        /**
         * Constructor of class
         */
        public function __construct()
        {
            // Init plugin default attributes
            $this->id = TOMAN_ESCROW_PAYMENT_GATEWAY_ID;
            $this->icon = apply_filters(TOMAN_ESCROW_PAYMENT_GATEWAY_ID . '_Logo', plugins_url('../../assets/images/logo.png', __FILE__)); // TODO: Route should be absolute
            $this->has_fields = false;
            $this->method_title = TOMAN_ESCROW_PLUGIN_TITLE;
            $this->method_description = TOMAN_ESCROW_PLUGIN_DESCRIPTION;

            // Method with all the options fields
            $this->init_form_fields();

            // Load the settings.
            $this->init_settings();

            // Fill title and description to show to the user (User can change it)
            $this->title = __('TomanPay Escrow Service Payment Method', 'toman-escrow-for-woocommerce');
            $this->description = 'در پرداخت امن تومن، مبلغ پرداختی نزد تومن به امانت می‌ماند و پس از انجام معامله در صورت تایید خریدار، تسویه با فروشنده صورت خواهد گرفت.';

            // TODO: Replace shopslug and passowrd with shop_slug and authentication_code
            $this->shopSlug = $this->settings['shopslug'];
            $this->authenticationCode = $this->settings['password'];

            // This action hook saves the settings
            add_action('woocommerce_update_options_payment_gateways_' . $this->id, [$this, 'process_admin_options']);

            // Define webhooks for returning from Toman Escrow app after user payment
            add_action('woocommerce_api_' . strtolower(get_class($this)), [$this, 'return_from_toman_gateway']);

            // Define action to redirect to user to the custom receipt page
            add_action('woocommerce_receipt_' . $this->id, [$this, 'redirect_to_receipt_page']);
        }

        /**
         * Plugin options, we deal with it in Step 3 too
         */
        public function init_form_fields()
        {
            $this->form_fields = [
                // TODO: Replace shopslug and passowrd with shop_slug and authentication_code
                'shopslug' => [
                    'title' => 'شناسه فروشگاه - Shop Slug',
                    'type' => 'text',
                    'description' => 'لطفا شناسه فروشگاه خود را وارد نمایید.'
                ],
                'password' => [
                    'title' => 'کد فعال‌سازی - Auth',
                    'type' => 'text',
                    'description' => 'لطفا کد فعال‌سازی را وارد نمایید.'
                ],
            ];
        }

        /**
         * Process payment
         *
         * @param $order_id
         * @return array
         */
        public function process_payment($order_id): array
        {
            $order = new WC_Order($order_id);

            return [
                'result' => 'success',
                'redirect' => $order->get_checkout_payment_url(true)
            ];
        }

        /**
         * Redirect to the receipt page and after that redirect to Toman webapp
         *
         * @param integer $orderId
         * @return void
         */
        public function redirect_to_receipt_page(int $orderId)
        {
            // Fetch order
            $order = new WC_Order($orderId);

            // Find currency rate
            $currencyRate = $this->get_currency_rate();

            // Find price details
            $priceDetails = $this->get_order_price_details($order, $currencyRate);

            // Find products details
            $productsDetails = $this->get_products_details($order, $currencyRate);

            // Get trace number from Toman API
            $traceNumber = $this->create_deal_in_toman($order, $productsDetails, $priceDetails);

            // Add trace number to the database
            $this->add_trace_number_to_order_in_database($order, $traceNumber);

            // Redirect to the Toman web app to pay
            $this->redirect_to_toman_webapp_to_pay($order, $traceNumber);
        }

        /**
         * Find currency rate
         *
         * @return integer
         */
        private function get_currency_rate(): int
        {
            $currency = get_woocommerce_currency();
            switch ($currency) {
                case 'IRR':
                    return 1;
                case 'IRT':
                    return 10;
                case 'IRHR':
                    return 1000;
                case 'IRHT':
                    return 10000;
                default:
                    return 1;
            }
        }

        /**
         * Find order price details
         *
         * @param object $order
         * @param integer $currencyRate
         * @return array
         */
        private function get_order_price_details(object $order, int $currencyRate): array
        {
            return [
                'total' => $order->get_total() * $currencyRate,
                'tax' => $order->get_total_tax() * $currencyRate,
                'discount' => $order->get_discount_total() * $currencyRate,
                'shipping' => $order->get_shipping_total() * $currencyRate,
            ];
        }

        /**
         * Get products details
         *
         * @param object $order
         * @param integer $currencyRate
         * @return array
         */
        private function get_products_details(object $order, int $currencyRate): array
        {
            $products = [];

            // Add products
            foreach ($order->get_items() as $product) {
                $product_name = $product->get_name();
                $description = null;
                if (strlen($product_name) > 150) {
                    $description = "نام کامل محصول: " . $product->get_name();
                    $product_name = substr($product_name, 0, 150);
                }
                $products[] = [
                    'name' => $product_name,
                    'quantity' => $product->get_quantity(),
                    'price' => (intval($product->get_total()) / $product->get_quantity()) * $currencyRate,
                    'description' => $description
                ];
            }

            // Add shipping info
            $shippingPrice = $order->get_shipping_total();
            if (intval($shippingPrice) > 0) {
                $products[] = [
                    'name' => 'هزینه ارسال',
                    'quantity' => 1,
                    'price' => intval($shippingPrice) * $currencyRate,
                ];
            }

            // Add tax
            $taxPrice = $order->get_total_tax();
            if (intval($taxPrice) > 0) {
                $products[] = [
                    'name' => 'مالیات',
                    'quantity' => 1,
                    'price' => intval($taxPrice) * $currencyRate,
                ];
            }

            // Add other fees
            $orderTotalPrice = $order->get_total() * $currencyRate;
            $itemsTotalPrice = 0;
            foreach ($products as $product) {
                $itemsTotalPrice += ($product['price'] * $product['quantity']);
            }
            $otherItemsTotalPrice = intval($orderTotalPrice - $itemsTotalPrice);
            if ($otherItemsTotalPrice > 0) {
                $products[] = [
                    'name' => 'هزینه‌های دیگر',
                    'quantity' => 1,
                    'price' => $otherItemsTotalPrice,
                ];
            }

            return $products;
        }

        /**
         * Get Toman API authorization header token
         *
         * @return string
         */
        public function get_toman_api_authorization_token(): string
        {
            return 'Basic ' . base64_encode($this->shopSlug . ':' . $this->authenticationCode);
        }

        /**
         * Create a new deal in Toman and it's return trace number
         *
         * @param object $order
         * @param array $products
         * @param array $priceDetails
         * @return void|string
         */
        private function create_deal_in_toman(object &$order, array $products, array $priceDetails)
        {
            // Prepare request
            $url = TOMAN_ESCROW_API_BASE_URL . '/users/me/shops/' . $this->shopSlug . '/deals';
            $body = [
                'res_number' => $order->get_id(),
                'items' => $products,
                'return_to' => add_query_arg(array('wc-api' => get_class($this), 'order_id' => $order->get_id()), get_site_url() . '/'),
                'price_details' => $priceDetails
            ];

            // Send request
            $response = wp_remote_post($url, [
                'body' => wp_json_encode($body),
                'headers' => $h = [
                    'Authorization' => $this->get_toman_api_authorization_token(),
                    'Plugin-Version' => TOMAN_PLUGIN_VERSION,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ],
                'sslverify' => false
            ]);

            // If response is fault, show error
            if (is_wp_error($response)) {
                // Add order note
                $order->add_order_note($response->get_error_message());

                // Redirect to checkout page
                $message = $response->get_error_message();
                $this->redirect_to_checkout_page_with_error($message);
            }

            // If trace number doesn't exist, show error
            $deal = json_decode($response['body']);
            if (!is_object($deal) || !isset($deal->trace_number)) {
                // Add order note
                $order->add_order_note("Trace number doesn't exist");

                // Redirect to checkout page
                $message = "Trace number doesn't exist";
                $this->redirect_to_checkout_page_with_error($message);
            }

            return $deal->trace_number;
        }

        public function get_funded_new_deal_count_in_toman()
        {
            // Prepare request
            $query_params = array(
                'state' => '(funded,new)',
                'page_size' => 1,
            );
            $url = TOMAN_ESCROW_API_BASE_URL . '/users/me/shops/' . $this->shopSlug . '/deals';
            $url = add_query_arg($query_params, $url);

            // Send request
            $response = wp_remote_get($url, [
                'headers' => $h = [
                    'Authorization' => $this->get_toman_api_authorization_token(),
                    'Plugin-Version' => TOMAN_PLUGIN_VERSION,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ],
                'sslverify' => false
            ]);

            // If response is fault, show error
            if (is_wp_error($response)) {
                return 0;
            }

            // If trace number doesn't exist, show error
            $result = json_decode($response['body']);
            if (!is_object($result) || !isset($result->count)) {
                return 0;
            }

            return $result->count;
        }

        /**
         * Add order trace number to post meta
         *
         * @param integer $orderId
         * @param string $traceNumber
         * @return void
         */
        private function add_trace_number_to_order_in_database(object &$order, string $traceNumber)
        {
            $order->update_meta_data('toman_escrow_trace_number', $traceNumber);
            $order->save();
        }

        /**
         * Redirect to Toman webapp to pay
         *
         * @param object $order
         * @param string $traceNumber
         * @return void
         */
        private function redirect_to_toman_webapp_to_pay(object &$order, string $traceNumber)
        {
            // Prepare request url
            $url = TOMAN_ESCROW_API_BASE_URL . '/deals/' . $traceNumber . '/redirect';

            // Redirect to the Toman web app
            try {
                session_write_close();
                echo wp_kses('<p>' . __("You're redirecting to TomanPay gateway to pay. It maybe take some seconds. Please wait ...", 'toman-escrow-for-woocommerce') . '</p>', array('p' => array()));
                echo '<a href="' . esc_url($url) . '">' . 'پرداخت' . '</a>';
                echo '<script>document.location = "' . esc_url($url) . '";</script>';
            } catch (Exception $exception) {
                $order->add_order_note($exception->getMessage());
                echo esc_html($exception->getMessage());
            }
        }

        /**
         * After return from Toman webapp (Gateway), check response, verify result by Toman API,
         * complete order, and finally redirect user to the 'order received' page
         *
         * @return void
         */
        public function return_from_toman_gateway()
        {
            // Check the response of callback after user payment 
            $this->check_callback_response_after_payment();// phpcs:ignore

            // Call Toman verify API and verify payment
            $traceNumber = $_POST['trace_number'];// phpcs:ignore
            $orderShouldBeCompleted = $this->verify_order_by_toman_api($traceNumber);

            // Complete order
            if ($orderShouldBeCompleted) {
                $order = new WC_Order($_POST['res_number']);// phpcs:ignore
                $this->complete_order($order, $traceNumber, true);

                // Redirect to the 'order-received' page
                $this->redirect_to_order_received_page_for_successfull_payments($order);
            }
        }

        /**
         * Check the response of callback after user payment
         *
         * @return void
         */
        private function check_callback_response_after_payment()
        {
            // Redirect to the checkout page with showing an error
            if (
                !isset($_POST['trace_number']) ||// phpcs:ignore
                !isset($_POST['state']) ||// phpcs:ignore
                !isset($_POST['sub_state']) ||// phpcs:ignore
                !isset($_POST['payable_amount']) ||// phpcs:ignore
                !isset($_POST['res_number']) ||// phpcs:ignore
                !isset($_POST['items_amount'])// phpcs:ignore
            ) {
                $message = __("You're payment failed. Please contact support", 'toman-escrow-for-woocommerce');
                $this->redirect_to_checkout_page_with_error($message);
            }

            // Check order existence
            $order = new WC_Order(sanitize_text_field($_POST['res_number']));// phpcs:ignore
            if (!$order) {
                $message = __("There is no order with this ID", 'toman-escrow-for-woocommerce');
                $this->redirect_to_checkout_page_with_error($message);
            }

            // Check status
            if (
                $_POST['state'] != WC_Toman_Safe_Pay::SUCCESS_PAYMENT_STATUS ||// phpcs:ignore
                $_POST['sub_state'] != WC_Toman_Safe_Pay::SUCCESS_PAYMENT_SUB_STATUS// phpcs:ignore
            ) {
                // $message = 'پرداخت با شکست مواجه شده است.';
                $message = __("You're payment failed", 'toman-escrow-for-woocommerce');
                $this->redirect_to_checkout_page_with_error($message);
            }

            // Check duplication payment
            if ($order->is_paid()) {
                $message = __('This order already has paid', 'toman-escrow-for-woocommerce');
                $order->add_order_note($message);
                wp_redirect($order->get_view_order_url());
                exit;
            }

            // Check order total amount
            if (intval($order->get_total() * $this->get_currency_rate()) != intval(sanitize_text_field($_POST['items_amount']))) {// phpcs:ignore
                $message = __('There is a conflict in order amount with paid amount', 'toman-escrow-for-woocommerce');
                $this->redirect_to_checkout_page_with_error($message);
            }
        }

        /**
         * Redirect to checkout page with error
         *
         * @param string $message
         * @return void
         */
        private function redirect_to_checkout_page_with_error(string $message)
        {
            wp_redirect(
                add_query_arg(
                    array('tipgmsg' => $message, 'tipgmt' => 'error'),
                    wc_get_checkout_url()
                )
            );
            exit;
        }

        /**
         * Verify order by Toman API
         *
         * @param string $traceNumber
         * @return bool|void
         */
        public function verify_order_by_toman_api(string $traceNumber)
        {
            $url = TOMAN_ESCROW_API_BASE_URL . '/users/me/shops/' . $this->shopSlug . '/deals/' . $traceNumber . '/verify';
            $response = wp_remote_request($url, [
                'method' => 'PATCH',
                'headers' => $h = [
                    'Authorization' => $this->get_toman_api_authorization_token(),
                    'Plugin-Version' => TOMAN_PLUGIN_VERSION,
                    'Content-Type' => 'application/json',
                    'Accept' => 'application/json'
                ],
                'sslverify' => false
            ]);

            // Fetch order
            $order = new WC_Order($_POST['res_number']);// phpcs:ignore

            // Check order existence
            if (!$order) {
                $message = __('There is no order with this specific', 'toman-escrow-for-woocommerce');
                $this->redirect_to_checkout_page_with_error($message);
            }

            // Check Toman has verified this order or not
            if (is_wp_error($response) || wp_remote_retrieve_response_code($response) != 204) {
                $message = __("We couldn't verify your payment. Please contact support", 'toman-escrow-for-woocommerce');
                $order->add_order_note($message);
                wp_redirect(wc_get_checkout_url());
                exit;
            }

            return true;
        }

        /**
         * Complete order and redirect to the order received page
         *
         * @param object $order
         * @param string $traceNumber
         * @return void
         */
        public function complete_order(object &$order, string $traceNumber, bool $should_empty_cart)
        {
            // Empty cart
            global $woocommerce;
            if ($should_empty_cart) {
                $woocommerce->cart->empty_cart();
            }

            // Update order status
            $order->update_status('wc-processing', __('Order is on progressing', 'toman-escrow-for-woocommerce'));

            // Add order note
            $message = __('Payment finished successfully', 'toman-escrow-for-woocommerce');
            $order->add_order_note($message);
        }

        /**
         * Redirect to received page after payment process of a successfull payment
         *
         * @param object $order
         * @return void
         */
        private function redirect_to_order_received_page_for_successfull_payments(object $order)
        {
            wp_redirect($this->get_return_url($order));
            exit;
        }

        /**
         * Sync last uncompleted orders statuses by Toman API
         *
         * @param object $orders
         * @return void
         */
        public function sync_last_uncompleted_orders_statuses_by_toman_api(&$orders)
        {
            // Get statuses from Toman API and update the database
            foreach ($orders as $wc_order) {
                // Get trace number from database
                $traceNumber = $wc_order->get_meta('toman_escrow_trace_number', true);
                if ($traceNumber == "") {
                    $traceNumber = $wc_order->get_meta('ultimate_toman_escrow_trace_number', true);
                    if ($traceNumber != "") {
                        $wc_order->update_meta_data('toman_escrow_trace_number', $traceNumber);
                        $wc_order->delete_meta_data('ultimate_toman_escrow_trace_number', $traceNumber);
                        $wc_order->save();
                    }
                }
                // Order doesn't have any trace number in the database
                if ($traceNumber == "") {
                    continue;
                }

                // Get order status from Toman API
                $url = TOMAN_ESCROW_API_BASE_URL . '/users/me/shops/' . $this->shopSlug . '/deals/' . $traceNumber;

                // Send request
                $response = wp_remote_get($url, [
                    'headers' => $h = [
                        'Authorization' => $this->get_toman_api_authorization_token(),
                        'Plugin-Version' => TOMAN_PLUGIN_VERSION,
                        'Content-Type' => 'application/json',
                        'Accept' => 'application/json'
                    ],
                    'sslverify' => false
                ]);

                // If response is fault, show error
                if (is_wp_error($response)) {
                    wp_die('Error in updating orders statuses');
                }

                // If trace number doesn't exist, show error
                $order = json_decode($response['body']);
                if (!is_object($order) || !isset($order->state) || !isset($order->sub_state)) {
                    continue;
                }

                // Get status based on state and sub_state
                $status = $this->get_order_status_based_on_state_and_sub_state_of_toman_api($order->state, $order->sub_state, $wc_order);

                // Update status
                if ($wc_order->get_status() != $status) {
                    $wc_order->update_status($status);
                }
            }
        }

        /**
         * Get order status based on defined state and sub_state in Toman
         *
         * @param string $state
         * @param string $subState
         * @param object $order
         * @return string|void
         */
        function get_order_status_based_on_state_and_sub_state_of_toman_api(string $state, string $subState, object &$order)
        {
            switch ($state) {
                case 'created':
                    switch ($subState) {
                        case 'seller_accepted':
                            $this->add_toman_status_description_to_order($order, 'در انتظار پرداخت');
                            return self::TOMAN_STATUS_PENDING;
                        case 'buyer_accepted':
                            $this->add_toman_status_description_to_order($order, 'در انتظار پرداخت');
                            return self::TOMAN_STATUS_PENDING;
                    }
                    break;
                case 'failed':
                    switch ($subState) {
                        case 'buyer_canceled':
                            $this->add_toman_status_description_to_order($order, 'لغو شده توسط خریدار');
                            return self::TOMAN_STATUS_CANCELLED;
                        case 'expired':
                            $this->add_toman_status_description_to_order($order, 'منقضی شده');
                            return self::TOMAN_STATUS_CANCELLED;
                        case 'seller_rejected':
                            $this->add_toman_status_description_to_order($order, 'لغو شده توسط فروشنده');
                            return self::TOMAN_STATUS_CANCELLED;
                        case 'shipment_expired':
                            $this->add_toman_status_description_to_order($order, 'لغو بدلیل عدم ارسال');
                            return self::TOMAN_STATUS_CANCELLED;
                        case 'sustained':
                            $this->add_toman_status_description_to_order($order, 'پایان یافته (ناموفق)');
                            return self::TOMAN_STATUS_CANCELLED;
                    }
                    break;
                case 'funded':
                    switch ($subState) {
                        case 'new':
                            $this->add_toman_status_description_to_order($order, 'در انتظار تایید خودکار فروشگاه');
                            return self::TOMAN_STATUS_PROCESSING;
                        case 'init':
                            $this->add_toman_status_description_to_order($order, 'در انتظار ارسال');
                            return self::TOMAN_STATUS_PROCESSING;
                        case 'sent':
                            $this->add_toman_status_description_to_order($order, 'در حال ارسال');
                            return self::TOMAN_STATUS_PROCESSING;
                        case 'leadtime_exhausted':
                            $this->add_toman_status_description_to_order($order, 'در انتظار تایید خریدار');
                            return self::TOMAN_STATUS_PROCESSING;
                    }
                    break;
                case 'success':
                    switch ($subState) {
                        case 'overruled':
                            $this->add_toman_status_description_to_order($order, 'پایان یافته (موفق)');
                            return self::TOMAN_STATUS_COMPLETED;
                        case 'buyer_accepted':
                            $this->add_toman_status_description_to_order($order, 'معامله موفق');
                            return self::TOMAN_STATUS_COMPLETED;
                        case 'autoaccept':
                            $this->add_toman_status_description_to_order($order, 'معامله موفق');
                            return self::TOMAN_STATUS_COMPLETED;
                    }
                    break;
                case 'jury':
                    $this->add_toman_status_description_to_order($order, 'شکایت خریدار');
                    return self::TOMAN_STATUS_ON_HOLD;
                default:
                    return null;
            }
        }

        /**
         * Add order status Toman description to post meta
         *
         * @param integer $orderId
         * @param string $description
         * @return void
         */
        function add_toman_status_description_to_order(object &$order, string $description)
        {
            $order->update_meta_data('toman_escrow_order_status_toman_description', $description);
            $order->save();
        }
    }
}

add_action('plugins_loaded', 'toman_escrow_define_plugin');
